-- Disable foreign key checks
SET FOREIGN_KEY_CHECKS = 0;

-- Drop existing views
DROP VIEW IF EXISTS customer_statements;
DROP VIEW IF EXISTS customer_pre_invoices;
DROP VIEW IF EXISTS customer_invoices;

-- Remove all foreign key constraints from orders table
ALTER TABLE orders
DROP FOREIGN KEY IF EXISTS orders_ibfk_1,
DROP FOREIGN KEY IF EXISTS orders_ibfk_2,
DROP FOREIGN KEY IF EXISTS orders_ibfk_3,
DROP FOREIGN KEY IF EXISTS orders_ibfk_4;

-- Drop existing tables in correct order (child tables first)
DROP TABLE IF EXISTS blanket_inventory;
DROP TABLE IF EXISTS preInvoices;
DROP TABLE IF EXISTS invoices;

-- Now we can safely drop the customers table
DROP TABLE IF EXISTS customers;

-- Create customers table
CREATE TABLE customers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    type ENUM('barn', 'stable', 'individual') NOT NULL,
    store_id VARCHAR(50),
    contact_name VARCHAR(100),
    email VARCHAR(100),
    phone VARCHAR(20),
    address VARCHAR(255),
    city VARCHAR(100),
    postal_code VARCHAR(20),
    number_of_stalls INT,
    facility_type VARCHAR(50),
    service_level VARCHAR(50),
    tack_room_cleaning BOOLEAN DEFAULT FALSE,
    feed_room_organization BOOLEAN DEFAULT FALSE,
    wash_stall_deep_clean BOOLEAN DEFAULT FALSE,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (store_id) REFERENCES stores(id)
);

-- Create blanket inventory table
CREATE TABLE blanket_inventory (
    id INT AUTO_INCREMENT PRIMARY KEY,
    customer_id INT,
    color VARCHAR(50) NOT NULL,
    type VARCHAR(50) NOT NULL,
    size VARCHAR(50),
    `condition` VARCHAR(50),
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (customer_id) REFERENCES customers(id)
);

-- Create preInvoices table
CREATE TABLE preInvoices (
    id INT AUTO_INCREMENT PRIMARY KEY,
    store_id VARCHAR(50),
    store_type VARCHAR(50) NOT NULL,
    week_number INT NOT NULL,
    total DECIMAL(10, 2) NOT NULL,
    status ENUM('draft', 'validated', 'approved', 'rejected') NOT NULL DEFAULT 'draft',
    customer_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (store_id) REFERENCES stores(id),
    FOREIGN KEY (customer_id) REFERENCES customers(id)
);

-- Create invoices table
CREATE TABLE invoices (
    id INT AUTO_INCREMENT PRIMARY KEY,
    pre_invoice_id INT,
    store_id VARCHAR(50),
    store_type VARCHAR(50) NOT NULL,
    month INT NOT NULL,
    year INT NOT NULL,
    total DECIMAL(10, 2) NOT NULL,
    status ENUM('draft', 'sent', 'paid', 'overdue', 'cancelled') NOT NULL DEFAULT 'draft',
    customer_id INT,
    due_date DATE,
    paid_date DATE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (pre_invoice_id) REFERENCES preInvoices(id),
    FOREIGN KEY (store_id) REFERENCES stores(id),
    FOREIGN KEY (customer_id) REFERENCES customers(id)
);

-- Modify orders table
ALTER TABLE orders
DROP COLUMN IF EXISTS customer_id,
DROP COLUMN IF EXISTS blanket_inventory_id,
DROP COLUMN IF EXISTS status;

ALTER TABLE orders
ADD COLUMN customer_id INT,
ADD COLUMN blanket_inventory_id INT,
ADD COLUMN status ENUM('draft', 'validated', 'statement', 'validated_statement', 'pre_invoice', 'invoice', 'archived') NOT NULL DEFAULT 'draft',
ADD FOREIGN KEY (customer_id) REFERENCES customers(id),
ADD FOREIGN KEY (blanket_inventory_id) REFERENCES blanket_inventory(id);

-- Create indexes
CREATE INDEX idx_customers_store_id ON customers(store_id);
CREATE INDEX idx_customers_type ON customers(type);
CREATE INDEX idx_orders_customer_id ON orders(customer_id);
CREATE INDEX idx_orders_status ON orders(status);
CREATE INDEX idx_preinvoices_store_id ON preInvoices(store_id);
CREATE INDEX idx_preinvoices_status ON preInvoices(status);
CREATE INDEX idx_blanket_inventory_customer_id ON blanket_inventory(customer_id);
CREATE INDEX idx_invoices_store_id ON invoices(store_id);
CREATE INDEX idx_invoices_status ON invoices(status);
CREATE INDEX idx_invoices_customer_id ON invoices(customer_id);

-- Migrate existing customer data
INSERT INTO customers (
    name, type, store_id, contact_name, email, phone, address, city, postal_code, created_at, updated_at
)
SELECT DISTINCT
    customer_name,
    CASE WHEN type = 'barn' THEN 'barn' ELSE 'individual' END,
    store_id,
    customer_name,
    customer_email,
    customer_phone,
    store_address,
    SUBSTRING_INDEX(SUBSTRING_INDEX(store_address, ',', 2), ',', -1),
    SUBSTRING_INDEX(store_address, ',', -1),
    MIN(saved_at),
    MIN(saved_at)
FROM orders
WHERE customer_name IS NOT NULL
GROUP BY customer_name, type, store_id, customer_email, customer_phone, store_address;

-- Update orders with customer_id
UPDATE orders o
JOIN customers c ON o.customer_name = c.name AND o.store_id = c.store_id
SET o.customer_id = c.id;

-- Create views
CREATE OR REPLACE VIEW customer_statements AS
SELECT 
    o.id as order_id,
    o.order_number,
    o.date,
    o.status,
    o.total,
    c.id as customer_id,
    c.name as customer_name,
    c.type as customer_type,
    c.contact_name,
    c.email as customer_email,
    c.phone as customer_phone,
    s.id as store_id,
    s.name as store_name,
    s.type as store_type,
    bi.id as blanket_inventory_id,
    bi.color as blanket_color,
    bi.type as blanket_type
FROM orders o
JOIN customers c ON o.customer_id = c.id
JOIN stores s ON o.store_id = s.id
LEFT JOIN blanket_inventory bi ON o.blanket_inventory_id = bi.id
WHERE o.status IN ('statement', 'validated_statement');

CREATE OR REPLACE VIEW customer_pre_invoices AS
SELECT 
    pi.id as pre_invoice_id,
    pi.store_id,
    pi.store_type,
    pi.week_number,
    pi.total,
    pi.status,
    pi.created_at,
    c.id as customer_id,
    c.name as customer_name,
    c.type as customer_type,
    c.contact_name,
    c.email as customer_email,
    c.phone as customer_phone,
    s.name as store_name
FROM preInvoices pi
JOIN customers c ON pi.customer_id = c.id
JOIN stores s ON pi.store_id = s.id;

CREATE OR REPLACE VIEW customer_invoices AS
SELECT 
    i.id as invoice_id,
    i.pre_invoice_id,
    i.store_id,
    i.store_type,
    i.month,
    i.year,
    i.total,
    i.status,
    i.due_date,
    i.paid_date,
    c.id as customer_id,
    c.name as customer_name,
    c.type as customer_type,
    c.contact_name,
    c.email as customer_email,
    c.phone as customer_phone,
    s.name as store_name
FROM invoices i
JOIN customers c ON i.customer_id = c.id
JOIN stores s ON i.store_id = s.id;

-- Re-enable foreign key checks
SET FOREIGN_KEY_CHECKS = 1; 