# backend/app/schemas/order.py
# Version: Definitive_Final

from pydantic import BaseModel, field_validator, Field
from typing import List, Optional, Any, Dict
from datetime import datetime, date
import json

# --- Schemas for Blanket Data (from Frontend) ---

class AdminDiscountSchemaIn(BaseModel):
    amount: float = 0
    type: str = '%'
    note: str = ''
    reason: str = ''

class BlanketDetailSchemaIn(BaseModel):
    id: str
    colors: List[str] = []
    isPlaid: bool = False
    hasWaterTreatment: bool = False
    hasHood: bool = False
    hasHeavySoil: bool = False
    hasHeavyHair: bool = False
    hasRepairs: bool = False
    repairMinutes: int = 0
    repairNotes: str = ''
    quantity: int = 1
    brand: str = ''
    hasPattern: bool = False
    patternNotes: str = ''
    foalMiniType: str = ''
    dogType: str = ''
    noteCategory: str = ''
    notes: str = ''
    adminNotes: str = ''
    basePrice: float = 0.0
    waterTreatmentPrice: float = 0.0
    blanketQualityColour: str = ''
    redFlag: bool = False
    redFlagNote: str = ''
    noWash: bool = False
    adminDiscount: AdminDiscountSchemaIn = Field(default_factory=AdminDiscountSchemaIn)

class BlanketSchemaIn(BaseModel):
    id: str
    type: str
    weight: str
    category: str
    details: List[BlanketDetailSchemaIn] = []


# --- Main Order Schemas ---

# Schema for CREATING or UPDATING an order (data coming IN to the API)
class OrderIn(BaseModel):
    id: Optional[str] = None
    store_id: str
    store_name: str
    customer_name: str
    customer_phone: Optional[str] = None
    customer_email: Optional[str] = None
    customer_address: Optional[str] = None
    customer_is_red_flagged: bool = False
    customer_red_flag_note: str = ''
    is_gh_staff_order: bool = False
    staff_name: Optional[str] = None
    week: Optional[str] = None
    status: str = 'draft'
    date: str
    total: float
    blankets: List[BlanketSchemaIn] = []
    order_colors: List[str] = []
    documents: Dict[str, Any] = {}
    orderNumber: Optional[str] = None


# Schema for Order Data going OUT of the API (the response)
class OrderOut(BaseModel):
    id: int
    order_number: str
    status: str
    date: date
    week: Optional[str] = None
    customer_name: str
    customer_phone: Optional[str] = None
    customer_email: Optional[str] = None
    customer_address: Optional[str] = None
    store_id: str
    store_name: str
    is_gh_staff_order: bool
    staff_name: Optional[str] = None
    total: float

    # These fields will be correctly parsed from JSON strings from the DB
    blankets: List[Any] = []
    order_colors: List[Any] = []
    documents: Dict[str, Any] = {}
    
    created_at: datetime
    updated_at: Optional[datetime] = None

    @field_validator('blankets', 'order_colors', 'documents', mode='before')
    @classmethod
    def parse_json_fields(cls, v: Any) -> Any:
        if isinstance(v, str):
            if not v: return [] if isinstance(v, list) else {}
            try: return json.loads(v)
            except json.JSONDecodeError: return []
        return v

    class Config:
        from_attributes = True
